<?php

/**
 * Mail folders modal view file.
 *
 * @package View
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Radosław Skrzypczak <r.skrzypczak@yetiforce.com>
 */

use App\Controller\ModalSettings;

/**
 * Mail folders modal view class.
 */
class Settings_OSSMailScanner_MailFoldersModal_View extends ModalSettings
{
	/** {@inheritdoc} */
	public $successBtn = 'LBL_SELECT_OPTION';

	/**
	 * @var bool Auto register events
	 */
	public $autoRegisterEvents = false;

	private ?Settings_OSSMailScanner_Record_Model $recordModel;

	/** {@inheritdoc} */
	public function checkPermission(App\Request $request)
	{
		parent::checkPermission($request);
		$this->recordModel = Settings_OSSMailScanner_Record_Model::getInstanceById($request->getInteger('template'));
		if (!$this->recordModel) {
			throw new \App\Exceptions\NoPermitted('LBL_PERMISSION_DENIED', 406);
		}
	}

	/**
	 * Tree in popup.
	 *
	 * @param \App\Request $request
	 */
	public function process(App\Request $request)
	{
		$moduleName = $request->getModule(false);
		$recordId = $request->getInteger('template');
		$missingFolders = $selectedFolders = $folders = [];
		$error = '';

		try {
			$mailAccount = \App\Mail\Account::getInstanceById($recordId);
			$imap = $mailAccount->openImap();
			$folders = $imap->getFolders();
		} catch (\Throwable $th) {
			\App\Log::error($th->__toString());
			$error = 'ERR_IMAP_CONNECTION';
			$this->successBtn = null;
		}

		$foldersFlat = array_unique(\App\Utils::flatten($folders));
		$mailScannerFolders = $this->recordModel->getFolders() ?: [];
		foreach ($mailScannerFolders as $folder) {
			if (!\in_array($folder, $foldersFlat)) {
				$missingFolders[] = $folder;
			} else {
				$selectedFolders[] = $folder;
			}
		}

		$viewer = $this->getViewer($request);
		$viewer->assign('TREE', \App\Json::encode($this->getTreeData($folders, $selectedFolders)));
		$viewer->assign('IS_MULTIPLE', true);
		$viewer->assign('MISSING_FOLDERS', $missingFolders);
		$viewer->assign('ERROR', $error);
		$viewer->view('Modals/MailFoldersModal.tpl', $moduleName);
	}

	/**
	 * Get tree item.
	 *
	 * @param array       $folder
	 * @param array       $selectedFolders
	 * @param array       $tree
	 * @param string|null $parent
	 *
	 * @return array
	 */
	public function getTreeItem(array $folder, array $selectedFolders, array &$tree, ?string $parent = null): array
	{
		$treeRecord = [
			'id' => $folder['fullName'],
			'type' => 'record',
			'parent' => $parent ?? '#',
			'text' => \App\Purifier::encodeHtml($folder['name']),
			'state' => ['selected' => \in_array($folder['fullName'], (array) $selectedFolders), 'opened' => true]
		];
		$treeRecord['db_id'] = $folder['fullName'];

		$tree[] = $treeRecord;
		if (!empty($folder['children'])) {
			foreach ($folder['children'] as $subfolder) {
				$this->getTreeItem($subfolder, $selectedFolders, $tree, $folder['fullName']);
			}
		}

		return $treeRecord;
	}

	/** {@inheritdoc} */
	public function getModalScripts(App\Request $request)
	{
		$jsFileNames = ['~libraries/jstree/dist/jstree.js'];
		$jsFileNames[] = '~layouts/resources/libraries/jstree.category.js';
		$jsFileNames[] = '~layouts/resources/libraries/jstree.checkbox.js';
		$jsFileNames[] = 'modules.Vtiger.resources.TreeModal';
		$jsFileNames[] = 'modules.Vtiger.resources.MailFoldersModal';

		return array_merge($this->checkAndConvertJsScripts($jsFileNames), parent::getModalScripts($request));
	}

	/** {@inheritdoc} */
	public function getModalCss(App\Request $request)
	{
		return array_merge(parent::getModalCss($request), $this->checkAndConvertCssStyles([
			'~libraries/jstree-bootstrap-theme/dist/themes/proton/style.css',
		]));
	}

	/** {@inheritdoc} */
	protected function preProcessTplName(App\Request $request)
	{
		return 'Modals/TreeHeader.tpl';
	}

	/**
	 * Get tree data.
	 *
	 * @param array $folders
	 * @param array $selectedFolders
	 *
	 * @return array
	 */
	private function getTreeData(array $folders, array $selectedFolders): array
	{
		$tree = [];
		foreach ($folders as $folder) {
			$this->getTreeItem($folder, $selectedFolders, $tree, null);
		}

		return $tree;
	}
}
