<?php
/**
 * Factory creates Field object instance
 *
 * @package App
 *
 * @copyright YetiForce S.A.
 * @license YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author Łukasz Krawczyk <l.krawczyk@yetiforce.com>
 */
declare(strict_types=1);

namespace App\Report\Factory\Field;

use App\FieldCoordinatorTransformer\QueryGeneratorFieldTransformer;
use App\Report\Factory\FieldFactory;
use App\Report\Model\Field;
use App\Report\Model\Field\YetiForceField;
use App\Report\Model\Field\YetiForceInventoryField;
use App\Report\Model\Query;
use Vtiger_Field_Model;
use Vtiger_Inventory_Model;
use Vtiger_Module_Model;

/** YetiForceFieldFactory class */
final class YetiForceFieldFactory implements FieldFactory
{
	/** {@inheritDoc} */
	public function create(Query $query, string $fieldCoordinates): ?Field
	{
		$fieldParts = QueryGeneratorFieldTransformer::extract($fieldCoordinates);
		$fieldName = $this->getFieldNameFromFieldPart($query, $fieldParts['fieldName']);
		$fieldModule = $fieldParts['moduleName'] ?? $query->getModuleName();

		if (true === QueryGeneratorFieldTransformer::isInventory($fieldCoordinates)) {
			return new YetiForceInventoryField(
				Vtiger_Inventory_Model::getInstance($fieldModule)->getField($fieldName),
			);
		}

		$fieldModel = Vtiger_Field_Model::getInstance(
			$fieldName,
			Vtiger_Module_Model::getInstance($fieldModule),
		);

		return false !== $fieldModel ? new YetiForceField($fieldModel) : null;
	}

	/** Get field name from field coordinates string */
	private function getFieldNameFromFieldPart(Query $query, string $needle): string
	{
		if (in_array($needle, $query->getStandardFields())) {
			return $needle;
		}

		$matchedCustomField = array_filter(
			$query->getCustomFields(),
			static fn ($customField): bool => $customField[1] === $needle,
		);
		$matchedCustomField = reset($matchedCustomField);

		return false === $matchedCustomField ? $needle : $matchedCustomField[0];
	}
}
