<?php
/**
 * Xls files generator class
 *
 * @package App
 *
 * @copyright YetiForce S.A.
 * @license YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author Leszek Koziatek <l.koziatek@yetiforce.com>
 */
declare(strict_types=1);

namespace App\DocumentBuilder\Generator;

use App\DocumentBuilder\FileGenerator;
use App\DocumentBuilder\FileGeneratorInterface;
use App\DocumentBuilder\Validator\XlsxValidator;
use App\DocumentBuilder\ValidatorInterface;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;

/** Xlsx file generator class */
final class XlsxGenerator implements FileGeneratorInterface
{
	/** @var string Default extension for created files */
	public const FILE_EXTENSION = 'xlsx';

	/** @var ValidatorInterface $validator */
	private ValidatorInterface $validator;

	public function __construct()
	{
		$this->validator = new XlsxValidator();
	}

	/**
	 * Generate Xlsx file
	 *
	 * @throws \Exception
	 */
	public function generate(string $filename, array $data, array $headers = [], array $groupedBy = []): void
	{
		$spreadsheet = new Spreadsheet();
		$this->validator->validate($data);

		$activeWorksheet = $spreadsheet->getActiveSheet();

		if ($headers) {
			$activeWorksheet->fromArray($headers);
		}

		$activeWorksheet->fromArray($data, null, $headers ? 'A2' : 'A1');
		foreach ($activeWorksheet->getColumnIterator() as $column) {
			$activeWorksheet->getColumnDimension($column->getColumnIndex())->setAutoSize(true);
		}
		$filepath = $this->getFilepath($filename);

		$writer = new Xlsx($spreadsheet);
		$writer->save($filepath);
	}

	/** Get filepath for file */
	public function getFilepath(string $filename): string
	{
		return sprintf('%s/%s.%s', FileGenerator::DOCUMENT_DIR, $filename, self::FILE_EXTENSION);
	}
}
