<?php
/**
 * CRM installation cli file.
 *
 * @package Cli
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Klaudia Łozowska <k.lozowska@yetiforce.com>
 */

namespace App\Cli;

use App\ConfigFile;
use App\Exceptions\AppException;
use App\Exceptions\IllegalValue;
use App\Exceptions\NoPermitted;
use App\Installer\Installer;
use App\Language;
use App\Module;
use App\Utils\ConfReport;
use App\Validator;
use yii\db\Exception;

/**
 * CRM installation cli class.
 */
class CrmInstallation extends Base
{
	/** @var array[] Database configuration arguments */
	private const DATABASE_ARGS = [
		'db_server' => [
			'prefix' => 'H',
			'description' => 'Database server',
			'type' => 'input',
			'allowEmpty' => false,
		],
		'db_port' => [
			'prefix' => 'P',
			'description' => 'Database port',
			'type' => 'input',
			'allowEmpty' => false,
		],
		'db_username' => [
			'prefix' => 'u',
			'description' => 'Database username',
			'type' => 'input',
			'allowEmpty' => false,
		],
		'db_password' => [
			'prefix' => 'p',
			'description' => 'Database password',
			'type' => 'password',
			'allowEmpty' => true,
		],
		'db_name' => [
			'prefix' => 'D',
			'description' => 'Database name',
			'type' => 'input',
			'allowEmpty' => false,
		],
	];

	/** @var array[] Mongo configuration arguments */
	private const MONGO_ARGS = [
		'mongo_server' => [
			'prefix' => 'mH',
			'description' => 'Mongo database server',
			'type' => 'input',
			'allowEmpty' => true,
		],
		'mongo_port' => [
			'prefix' => 'mP',
			'description' => 'Mongo database port',
			'type' => 'input',
			'allowEmpty' => true,
		],
		'mongo_username' => [
			'prefix' => 'mu',
			'description' => 'Mongo database username',
			'type' => 'input',
			'allowEmpty' => true,
		],
		'mongo_password' => [
			'prefix' => 'mp',
			'description' => 'Mongo database password',
			'type' => 'password',
			'allowEmpty' => true,
		],
		'mongo_name' => [
			'prefix' => 'mD',
			'description' => 'Mongo database name',
			'type' => 'input',
			'allowEmpty' => true,
		],
	];

	/** @var array[] Main configuration arguments */
	private const MAIN_CONFIG_ARGS = [
		'site_URL' => [
			'prefix' => 's',
			'description' => 'Site URL (with protocol and slash on the end)',
			'type' => 'input',
			'allowEmpty' => false,
		],
		'default_timezone' => [
			'prefix' => 'T',
			'description' => 'Timezone',
			'type' => 'radio',
			'allowEmpty' => false,
		],
	];

	/** @var array[] Admin configuration arguments */
	private const ADMIN_ARGS = [
		'user_name' => [
			'prefix' => 'A',
			'description' => 'Admin username',
			'type' => 'input',
			'allowEmpty' => false,
		],
		'password' => [
			'prefix' => 'Ap',
			'description' => 'Admin password',
			'type' => 'password',
			'allowEmpty' => false,
		],
		'first_name' => [
			'prefix' => 'Af',
			'description' => 'Admin first name',
			'type' => 'input',
			'allowEmpty' => false,
		],
		'last_name' => [
			'prefix' => 'Al',
			'description' => 'Admin last name',
			'type' => 'input',
			'allowEmpty' => false,
		],
		'email' => [
			'prefix' => 'Ae',
			'description' => 'Admin email address',
			'type' => 'input',
			'allowEmpty' => false,
		],
		'currency' => [
			'prefix' => 'c',
			'description' => 'Currency',
			'type' => 'radio',
			'allowEmpty' => false,
		],
		'date_format' => [
			'prefix' => 'df',
			'description' => 'Date format',
			'type' => 'radio',
			'allowEmpty' => false,
		],
		'business_profile' => [
			'prefix' => 'bp',
			'description' => 'Business profile',
			'type' => 'checkboxes',
			'allowEmpty' => false,
		],
	];

	/** {@inheritdoc} */
	protected string $moduleName = 'System';

	/** @var string[] Methods list */
	protected array $methods = [
		'install' => 'Install YetiForce',
	];

	/** @var Installer Installer. */
	private Installer $installer;

	/** @var string|null Language. */
	private ?string $language = null;

	/** @var bool No interaction mode. */
	private bool $noInteraction = false;

	/**
	 * Install CRM.
	 *
	 * @throws IllegalValue
	 * @throws AppException
	 * @throws Exception
	 * @throws \ReflectionException
	 *
	 * @return void
	 */
	public function install(): void
	{
		$this->setNoInteractionMode();

		$this->installer = new Installer();
		if ($this->installer->isInstalled()) {
			$this->climate->lightGreen('CRM is already installed :)');
			$this->returnToActionList();
		}

		$this->setLanguage();

		$this->climate->br();

		if (!$this->noInteraction && !$this->getLicenceAgreement()) {
			$this->climate->red('Licence must be accepted.');
			$this->returnToActionList();
		}

		$this->climate->br();

		if (!$this->checkSystemRequirements()) {
			$this->returnToActionList();
		}

		$this->climate->br();

		if (!$this->installer->isDbInitialized()) {
			$this->createDbConfigFile();
			$this->initializeDb();
		}

		$this->updateMainConfigFile();
		$this->createConfigFiles();

		$this->prepareAdminUser();

		$this->installer->setLanguage($this->language);
		$this->installer->updateVersion();

		chmod(ROOT_DIRECTORY . '/cron/cron.sh', 0744);
	}

	/**
	 * Set no-interaction execution mode.
	 *
	 * @return void
	 */
	private function setNoInteractionMode(): void
	{
		$this->climate->arguments->add([
			'noInteraction' => [
				'prefix' => 'n',
				'description' => 'No interaction mode',
				'noValue' => true,
			],
		]);
		$this->climate->arguments->parse();

		if ($this->climate->arguments->defined('noInteraction')) {
			$this->noInteraction = true;
		}
	}

	/**
	 * Set language.
	 *
	 * @return void
	 */
	private function setLanguage(): void
	{
		$this->climate->arguments->add([
			'lang' => [
				'prefix' => 'l',
				'description' => 'System language',
			],
		]);
		$this->climate->arguments->parse();

		$languages = array_map(static fn (array $lang) => $lang['displayName'], Installer::getLanguages());
		asort($languages);
		$languages = array_merge([Language::DEFAULT_LANG => 'English (United States)'], $languages);

		if ($this->climate->arguments->defined('lang')) {
			$language = $this->climate->arguments->get('lang');

			if (!\array_key_exists($language, $languages)) {
				$this->climate->red(sprintf(
					'Provided language is invalid. Available values: %s',
					implode(', ', array_keys($languages))
				));

				$this->returnToActionList();
			}
		} elseif ($this->noInteraction) {
			$this->missingArgument('lang');

			return;
		} else {
			$language = $this->climate->radio('Choose language:', $languages)->prompt();
		}

		$this->language = $language;
	}

	/**
	 * Check if license is accepted.
	 *
	 * @return bool
	 */
	private function getLicenceAgreement(): bool
	{
		$license = 'pl-PL' === $this->language ? 'LicensePL' : 'LicenseEN';
		$license = "See the license in here: ./licenses/$license.txt";

		$this->climate->out($license);

		return $this->climate->confirm(
			'Please read the license for YetiForce system. Do you agree with its conditions?'
		)->confirmed();
	}

	/**
	 * Check system requirements.
	 *
	 * @throws IllegalValue
	 *
	 * @return bool
	 */
	private function checkSystemRequirements(): bool
	{
		$requirements = ConfReport::getByType(array_keys(Installer::SYS_REQS_CHECK));

		$unmetRequirements = [];
		foreach ($requirements as $type => $reqs) {
			$unmetRequirements[$type] = array_filter(
				$reqs,
				static fn (array $item) => !$item['status']
					&& (empty($item['mode']) || \in_array($item['mode'], ['showWarnings', 'showErrors'])),
			);
		}

		$this->climate->bold('System requirements');

		if (!empty($unmetRequirements = array_filter($unmetRequirements))) {
			$this->climate->red('Some system requirements are not being met!');

			foreach ($unmetRequirements as $type => $items) {
				$this->climate->red(Installer::SYS_REQS_CHECK[$type] . ':');
				$this->climate->backgroundRed()->table($this->getRequirementsTableData($items, $type));
				$this->climate->br();
			}
		} else {
			$this->climate->green('Your system meets all the requirements. Good job!');
		}

		return $this->noInteraction || $this->navigateSysReqsMenu($requirements, !empty($unmetRequirements));
	}

	/**
	 * Prepare system requirements display data.
	 *
	 * @param array  $items
	 * @param string $type
	 *
	 * @return array
	 */
	private function getRequirementsTableData(array $items, string $type): array
	{
		$data = [];
		foreach ($items as $name => $item) {
			if ('libraries' === $type) {
				$data[] = [
					'Support for libraries' => $name,
					'Mandatory' => $item['mandatory'] ?? false ? 'Mandatory' : 'Optional',
					'Installed' => 'LBL_YES' === $item['www'] ? 'Yes' : 'No',
				];
			} elseif (\in_array($type, ['security', 'stability', 'performance'])) {
				$recommended = preg_replace(
					'/<b class="text-danger">(\w+)<\/b>/',
					'${1}',
					$item['recommended'] ?? ''
				);

				$data[] = [
					'Parameter' => $name,
					'Recommended' => $recommended,
					'Present values' => $item['www'] ?? null,
				];
			} elseif ('publicDirectoryAccess' === $type) {
				$data[] = [
					'Directory path' => $name,
					'Is access denied?' => $item['status'] ?? null,
				];
			} elseif ('environment' === $type) {
				$data[] = [
					'Parameter' => $name,
					'Present values' => $item['www'] ?? null,
				];
			} elseif ('writableFilesAndFolders' === $type) {
				$data[] = [
					'Path' => $name,
					'Permission' => 'LBL_YES' === $item['www'] ? 'Yes' : 'No',
				];
			}
		}

		return $data;
	}

	/**
	 * Navigate through system requirements check.
	 *
	 * @param array $requirements
	 * @param bool  $errors
	 *
	 * @return bool
	 */
	private function navigateSysReqsMenu(array $requirements, bool $errors = false): bool
	{
		$actions = [
			'proceed' => 'Proceed',
			'displayReqs' => 'Display system requirements',
			'abort' => 'Abort',
		];
		$next = $this->climate->radio('Choose action:', $actions)->prompt();

		if ('displayReqs' === $next) {
			$displayActions = array_merge(Installer::SYS_REQS_CHECK, ['return' => 'Return']);
			$type = $this->climate->radio('Choose section:', $displayActions)->prompt();

			if ('return' === $type) {
				return $this->navigateSysReqsMenu($requirements, $errors);
			}

			$this->climate->table($this->getRequirementsTableData($requirements[$type], $type));

			return $this->navigateSysReqsMenu($requirements, $errors);
		}
		if ('proceed' === $next) {
			if ($errors) {
				return $this->climate->red()->confirm(
					'Some settings do not meet the recommended values. This may cause incorrect system installation or subsequent errors in the YetiForce system operation. Are you sure you want to continue?'
				)->confirmed();
			}

			return true;
		}
		if ('abort' === $next) {
			return false;
		}

		$this->climate->red('Invalid option. Abort...');

		return false;
	}

	/**
	 * Display missing argument message.
	 *
	 * @param string $arg
	 *
	 * @return void
	 */
	private function missingArgument(string $arg): void
	{
		$arg = $this->climate->arguments->all()[$arg];

		$this->climate->red(sprintf('[no-interaction] Missing argument: %s [-%s]', $arg->name(), $arg->prefix()));

		$this->returnToActionList();
	}

	/**
	 * Create config files.
	 *
	 * @throws AppException
	 * @throws IllegalValue
	 */
	private function createConfigFiles(): void
	{
		$this->climate->green('Creating config files...');

		$this->installer->createConfigFiles();
	}

	/**
	 * Initialize database.
	 *
	 * @throws AppException
	 * @throws IllegalValue
	 *
	 * @return void
	 */
	private function createDbConfigFile(): void
	{
		$configFile = new ConfigFile('db');

		$this->climate->info('Configuring database...');

		$first = true;
		while (!Installer::checkDbConnection($configFile->getData())['flag']) {
			$this->updateConfigFile($configFile, self::DATABASE_ARGS, !$first);
			$first = false;

			if (Installer::checkDbConnection($configFile->getData())['flag']) {
				$configFile->create();
			} elseif ($this->noInteraction) {
				$this->climate->red('Database configuration failed.');
				$this->returnToActionList();
			} else {
				$this->climate->red('Invalid configuration.');
			}
		}

		$first = true;
		if (\extension_loaded('mongodb')) {
			$this->climate->info('Configuring Mongo...');
			while (!Installer::checkMongoConnection($configFile->getData())['flag']) {
				$this->updateConfigFile($configFile, self::MONGO_ARGS, !$first);
				$first = false;

				if (Installer::checkMongoConnection($configFile->getData())['flag']) {
					$configFile->create();
				} elseif ($this->noInteraction) {
					$this->climate->red('Mongo configuration failed. Skipping.');

					break;
				} else {
					$action = $this->climate
						->red()
						->confirm('Mongo configuration failed. Do you want to reconfigure? Press N to proceed.');
					if (!$action->confirmed()) {
						break;
					}
				}
			}
		}
	}

	/**
	 * Update main config file.
	 *
	 * @throws AppException
	 * @throws IllegalValue
	 *
	 * @return void
	 */
	private function updateMainConfigFile(): void
	{
		$configFile = new ConfigFile('main');

		$this->climate->info('Main configuration');

		$args = self::MAIN_CONFIG_ARGS;
		$args['default_timezone']['options'] = \UserTimeZones::getTimeZones();

		$this->updateConfigFile($configFile, $args);
		$configFile->create();
	}

	/**
	 * Set DB configuration.
	 *
	 * @param ConfigFile $configFile
	 * @param array      $args
	 * @param bool       $skipArgs
	 *
	 * @throws AppException
	 *
	 * @return void
	 */
	private function updateConfigFile(ConfigFile $configFile, array $args, bool $skipArgs = false): void
	{
		$this->climate->arguments->add($args);
		$this->climate->arguments->parse();

		foreach ($args as $name => $argument) {
			if (!$skipArgs && $this->climate->arguments->defined($name)) {
				$value = $this->climate->arguments->get($name);
			} elseif ($this->noInteraction) {
				if ($argument['allowEmpty']) {
					continue;
				} else {
					$this->missingArgument($name);

					return;
				}
			} else {
				$value = $this->getArgValue($argument);
			}

			if (!$value && $argument['allowEmpty']) {
				continue;
			}

			do {
				try {
					$configFile->set($name, $value);
					$valid = true;
				} catch (IllegalValue $e) {
					if ($this->noInteraction) {
						$this->climate->red(sprintf(
							'Invalid %s [-%s].',
							lcfirst($argument['description']),
							$argument['prefix']
						));
						if (\in_array($argument['type'], ['radio', 'checkboxes'])) {
							$this->climate->red(sprintf(
								'Available values: %s',
								implode(', ', $argument['options'])
							));
						}

						$this->returnToActionList();
					}

					$this->climate->red('Invalid value.');
					$value = $this->getArgValue($argument);
					$valid = false;
				}
			} while (!$valid);
		}
	}

	/**
	 * Ask for value.
	 *
	 * @param array $arg
	 *
	 * @return string|array
	 */
	private function getArgValue(array $arg): string|array
	{
		do {
			if (\in_array($arg['type'], ['radio', 'checkboxes'])) {
				if (!($arg['options'] ?? null)) {
					$this->climate->red('Missing options.');
					$this->returnToActionList();
				}
				$value = $this->climate->{$arg['type']}($arg['description'] . ':', $arg['options'])->prompt();
			} else {
				$value = $this->climate->{$arg['type']}($arg['description'] . ':')->prompt();
			}

			if ('password' === $arg['type']) {
				$valueRetype = $this->climate->{$arg['type']}('Retype ' . lcfirst($arg['description']) . ':')->prompt();
				if ($valueRetype !== $value) {
					$this->climate->red('This values don\'t match!');
					$value = null;

					continue;
				}
			}

			if (!$arg['allowEmpty'] && empty($value)) {
				$this->climate->red('This value is required!');
			}
		} while (!$arg['allowEmpty'] && empty($value));

		return $value;
	}

	/**
	 * Initialize database.
	 *
	 * @return void
	 */
	private function initializeDb(): void
	{
		$this->climate->green('Migrating database...');

		try {
			$this->installer->getMigrator()->migrate();
		} catch (\Throwable $e) {
			$this->climate->red(sprintf('Migrating database failed. Reason: %s', $e->getMessage()));

			$this->returnToActionList();
		}
	}

	/**
	 * Prepare admin user.
	 *
	 * @throws Exception
	 * @throws \ReflectionException
	 * @throws NoPermitted
	 */
	private function prepareAdminUser(): void
	{
		$businessProfiles = array_map(static fn (array $profile) => $profile['name'], Module::getPackages());
		$currencies = array_flip(
			array_map(static fn (array $currency) => $currency[0], Installer::getCurrencies())
		);

		$this->climate->green('Creating admin user...');

		$args = self::ADMIN_ARGS;
		$args['currency']['options'] = $currencies;
		$args['date_format']['options'] = Installer::DATE_FORMATS;
		$args['business_profile']['options'] = $businessProfiles;

		$this->climate->arguments->add($args);
		$this->climate->arguments->parse();

		$data = [];
		foreach ($args as $name => $argument) {
			if ($this->climate->arguments->defined($name)) {
				$data[$name] = $this->climate->arguments->get($name);

				if ('checkboxes' === $argument['type']) {
					$data[$name] = explode(',', $data[$name]);
				}
			} elseif ($this->noInteraction) {
				if ($argument['allowEmpty']) {
					continue;
				} else {
					$this->missingArgument($name);

					return;
				}
			} else {
				$data[$name] = $this->getArgValue($argument);
			}

			do {
				$valid = true;
				switch ($name) {
					case 'user_name':
						$blacklist = require ROOT_DIRECTORY . '/config/username_blacklist.php';

						if (\in_array($data[$name], $blacklist) || !preg_match('/^[a-zA-Z0-9_.@-]{3,64}$/', $data[$name])) {
							$valid = false;
						}

						break;
					case 'currency':
						if (!\array_key_exists($data[$name], $currencies)) {
							$valid = false;
						}

						break;
					case 'date_format':
						if (!\in_array($data[$name], Installer::DATE_FORMATS)) {
							$valid = false;
						}

						break;
					case 'email':
						if (!Validator::email($data[$name])) {
							$valid = false;
						}

						break;
					case 'password':
						if ($error = \Settings_Password_Record_Model::checkPassword($data[$name])) {
							$this->climate->red($error);
							$valid = false;
						}

						break;
					case 'business_profile':
						if (!empty(array_diff($data[$name], $businessProfiles))) {
							$valid = false;
						}

						break;
					default:
						break;
				}

				if (!$valid) {
					if ($this->noInteraction) {
						$this->climate->red(sprintf(
							'Invalid %s [-%s].',
							lcfirst($argument['description']),
							$argument['prefix']
						));
						if (\in_array($argument['type'], ['radio', 'checkboxes'])) {
							$this->climate->red(sprintf(
								'Available values: %s',
								implode(', ', $argument['options'])
							));
						}

						$this->returnToActionList();
					}

					$this->climate->red('Invalid value.');
					$data[$name] = $this->getArgValue($argument);
				}
			} while (!$valid || !($data[$name] || $argument['allowEmpty']));
		}

		$this->installer->setCurrency($data['currency']);
		try {
			$this->installer->prepareAdminUser(
				$data['user_name'],
				$data['password'],
				$data['email'],
				$data['first_name'],
				$data['last_name'],
				$data['date_format'],
			);
		} catch (\Throwable $e) {
			$this->climate->red('Admin user could not be updated. Reason: ' . $e->getMessage());
		}

		$this->installer->recalculateSharingRules();

		$this->adjustModules($data['business_profile']);
	}

	/**
	 * Adjust modules to business profile.
	 *
	 * @param string[] $profiles
	 *
	 * @return void
	 */
	private function adjustModules(array $profiles): void
	{
		$this->climate->green('Enabling modules...');

		$modulePackageIds = array_map(
			static fn (array $package) => $package['id'],
			array_filter(
				Module::getPackages(),
				static fn (array $package) => \in_array($package['name'], $profiles),
			)
		);

		$this->installer::adjustModules($modulePackageIds);
	}
}
